﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Data.Linq;
using VeteransAffairs.Registries.BusinessManager.Utilities;
using VeteransAffairs.Registries.BusinessAHOBPR;
using System.Reflection;
using System.Web.Script.Serialization;

namespace VeteransAffairs.Registries.BusinessManagerAHOBPR
{
    public class AHOBPRFormResponseManager : AHOBPRBaseBO 
    {
        private AHOBPRShared _sharedManager = new AHOBPRShared();

        #region Reset global variables
        /// <summary>
        /// Reset global variables for form lookups
        /// </summary>
        /// <returns></returns>
        public bool ResetGlobalVariables()
        {
            AHOBPRGlobal.FormLookup = null;
            AHOBPRGlobal.FormSectionLookup = null;
            AHOBPRGlobal.FormSubSectionLookup = null;
            AHOBPRGlobal.FormSectionLookup = null;
            AHOBPRGlobal.FormAnswerLookup = null;
            return true;
        }
        #endregion
        #region Get Form related IDs

        /// <summary>
        /// Get Form ID by BPR Form ID
        /// </summary>
        /// <param name="bprFormId"></param>
        /// <returns></returns>
        public int GetFormId(string bprFormId)
        {
            int id = 0;
            if (string.IsNullOrEmpty(bprFormId))
            {
                return id;
            }
            else
            {
                if (AHOBPRGlobal.FormLookup == null)
                {
                    using (_dbAhobpr = GetDataContext())
                    {
                        AHOBPRGlobal.FormLookup = (from e in _dbAhobpr.STD_FORMs
                                                   select e).ToList();
                    }
                }
                
                try
                {
                    id = (from e in AHOBPRGlobal.FormLookup
                                    where e.BPR_FORM_ID == bprFormId
                                    select e.STD_FORM_ID).FirstOrDefault();
                }
                catch
                {
                    id = 0;
                }

               return id;
            }
       }

        public int GetFollowupFormId(string bprFormId)
        {
            AHOBPRGlobal.FormLookup = null;
            AHOBPRGlobal.FormQuestionLookup = null;
            return GetFormId(bprFormId);
        }

   
        /// <summary>
        /// Update REQUEST_SEEN_RESPONSE and HEALTH_CONCERNS_RESPONSE column in REGISTRANT table
        /// </summary>
        /// <param name="registrantId"></param>
        /// <returns></returns>
        public int UpdateRegistrantWithResponseInfo(int registrantId)
        {
            using (_dbAhobpr = GetDataContext())
            {
                return _dbAhobpr.SP_UpdateRegistrantResponse(registrantId);
            }
        }


        /// <summary>
        /// Remove Questionnaire Complete Date when the status is changed from PARTICIPANT to REVIEWED-ELIGIBLE
        /// </summary>
        /// <param name="userId"></param>
        public void RemoveQuestionnaireCompleteDate(string userId)
        {
            using (_dbAhobpr = GetDataContext())
            {
                try
                {
                 _dbAhobpr.SP_RemoveQuestionnaireCompleteDate(userId);
                }
                catch (Exception ex)
                {
                    _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                }
            }
        }

        /// <summary>
        /// Get Section ID by formID and Bpr Section Id
        /// </summary>
        /// <param name="formId"></param>
        /// <param name="bprSectionId"></param>
        /// <returns></returns>
        public int GetSectionId(int formId, int bprSectionId)
        {
            int id = 0;
            if (formId == 0 || bprSectionId == 0)
            {
                return id;
            }
            else
            {
                if (AHOBPRGlobal.FormSectionLookup == null)
                {
                    using (_dbAhobpr = GetDataContext())
                    {
                        AHOBPRGlobal.FormSectionLookup = (from e in _dbAhobpr.STD_FORM_SECTIONs
                                                   select e).ToList();
                    }
                }

                try
                {
                    id = (from e in AHOBPRGlobal.FormSectionLookup
                            where e.STD_FORM_ID == formId && e.BRP_FORM_SECTION_ID == bprSectionId
                            select e.STD_FORM_SECTION_ID).FirstOrDefault();
                }
                catch
                {
                    id = 0;
                }

                return id;
            }
        }


        /// <summary>
        /// Get SubSection ID by sectionId and bprSubSectionId
        /// </summary>
        /// <param name="sectionId"></param>
        /// <param name="bprSubSectionId"></param>
        /// <returns></returns>
        public int GetSubSectionId(int formId, int sectionId, int bprSubSectionId)
        {
            int id = 0;
            if (sectionId == 0 || bprSubSectionId == 0)
            {
                return id;
            }
            else
            {
                if (AHOBPRGlobal.FormSubSectionLookup == null)
                {
                    using (_dbAhobpr = GetDataContext())
                    {
                        AHOBPRGlobal.FormSubSectionLookup = (from e in _dbAhobpr.STD_FORM_SUB_SECTIONs
                                                          select e).ToList();
                    }
                }
                
                try
                {
                    id = (from e in AHOBPRGlobal.FormSubSectionLookup
                            where e.STD_FORM_ID == formId && e.STD_FORM_SECTION_ID == sectionId
                                && e.BRP_FORM_SUB_SECTION_ID == bprSubSectionId
                            select e.STD_FORM_SUB_SECTION_ID).FirstOrDefault();
                }
                catch
                {
                    id = 0;
                }
  
                return id;
            }
        }

        /// <summary>
        /// Get DeploymentId by regsitrant ID and bprDeploymentId
        /// </summary>
        /// <param name="registrantId"></param>
        /// <param name="bprDeploymentId"></param>
        /// <returns></returns>
        public int GetDeploymentId(int registrantId, string bprDeploymentId)
        {
            int id = 0;
            if (string.IsNullOrEmpty(bprDeploymentId))
            {
                return id;
            }
            else
            {
                using (_dbAhobpr = GetDataContext())
                {
                   id = (from e in _dbAhobpr.REGISTRANT_DEPLOYMENTs
                         where e.REGISTRANT_ID == registrantId && e.BPR_DEPLOYMENT_ID == bprDeploymentId
                         select e.REGISTRANT_DEPLOYMENT_ID).FirstOrDefault();
                }
            }
            return id;
        }

        public List<string> GetAllBprDeploymentIdsbyRegistrantId(int registrantId)
        {
            List<string> deploymentIds = new List<string>();

            if (registrantId == 0)
            {
                return deploymentIds;
            }
            else
            {
                using (_dbAhobpr = GetDataContext())
                {
                    deploymentIds = (from e in _dbAhobpr.REGISTRANT_DEPLOYMENTs
                          where e.REGISTRANT_ID == registrantId 
                          select e.BPR_DEPLOYMENT_ID).ToList();
                }
            }
            return deploymentIds;
        }

        /// <summary>
        /// Get STD_FORM_QUESTION_ID by bprSectionId, bprSubSectionId and bprQuestionId
        /// </summary>
        /// <param name="formId"></param>
        /// <param name="bprSectionId"></param>
        /// <param name="bprSubSectionId"></param>
        /// <param name="bprQuestionId"></param>
        /// <returns></returns>
        public int GetFormQuestionId(int formId, int bprSectionId, int bprSubSectionId, string bprQuestionId)
        {
            int id = 0;
            if (formId == 0 || bprSectionId == 0 || bprSubSectionId == 0 || string.IsNullOrEmpty(bprQuestionId))
            {
                return id;
            }
            else
            {
                if (AHOBPRGlobal.FormQuestionLookup == null)
                {
                    using (_dbAhobpr = GetDataContext())
                    {
                        AHOBPRGlobal.FormQuestionLookup = (from e in _dbAhobpr.STD_FORM_QUESTIONs
                                                             select e).ToList();
                    }
                }

                try
                {
                    string uniqueQuestionId = GetUniqueQuestionNumber(formId, bprSectionId, bprSubSectionId, bprQuestionId);

                    id = (from e in AHOBPRGlobal.FormQuestionLookup
                          where e.STD_FORM_ID == formId && e.FORM_QUESTION_UNIQUE_ID == uniqueQuestionId
                          select e.STD_FORM_QUESTION_ID).FirstOrDefault();
                }
                catch
                {
                    id = 0;
                }

                return id;
            }
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="formId"></param>
        /// <param name="bprSectionId"></param>
        /// <param name="bprSubSectionId"></param>
        /// <param name="bprQuestionId"></param>
        /// <returns></returns>
        public int GetFormQuestionIdForFollowupQuestion(int formId, string bprQuestionId)
        {
            int id = 0;
            if (formId == 0 || string.IsNullOrEmpty(bprQuestionId))
            {
                return id;
            }
            else
            {
                if (AHOBPRGlobal.FormQuestionLookup == null)
                {
                    using (_dbAhobpr = GetDataContext())
                    {
                        AHOBPRGlobal.FormQuestionLookup = (from e in _dbAhobpr.STD_FORM_QUESTIONs
                                                           select e).ToList();
                    }
                }

                try
                {
                     id = (from e in AHOBPRGlobal.FormQuestionLookup
                          where e.STD_FORM_ID == formId && e.FORM_QUESTION_UNIQUE_ID == bprQuestionId
                          select e.STD_FORM_QUESTION_ID).FirstOrDefault();
                }
                catch
                {
                    id = 0;
                }

                return id;
            }
        }

        /// <summary>
        /// Get unique question number by bprSectionId, bprSubSectionId and bprQuestionId
        /// </summary>
        /// <param name="formId"></param>
        /// <param name="bprSectionId"></param>
        /// <param name="bprSubSectionId"></param>
        /// <param name="bprQuestionId"></param>
        /// <returns></returns>
        private string GetUniqueQuestionNumber(int formId, int bprSectionId, int bprSubSectionId, string bprQuestionId)
        {
            string uniqueId = string.Empty;

            if (HasMoreThanOneSubsection(formId, bprSectionId) || bprSectionId == 4)
            {
                uniqueId = bprSectionId.ToString() + "." + bprSubSectionId.ToString() + "." + bprQuestionId;
            }
            else
            {
                uniqueId = bprSectionId.ToString() + "." + bprQuestionId;
            }

            return uniqueId;
        }

        /// <summary>
        /// Check if a section has more than 1 subsection
        /// </summary>
        /// <param name="formId"></param>
        /// <param name="bprSectionId"></param>
        /// <returns></returns>
        private bool HasMoreThanOneSubsection(int formId, int bprSectionId)
        {
            bool result = false;

            int sqlSectionId = GetSectionId(formId, bprSectionId);
            if (sqlSectionId > 0)
            {
                if (AHOBPRGlobal.FormSubSectionLookup == null)
                {
                    using (_dbAhobpr = GetDataContext())
                    {
                        AHOBPRGlobal.FormSubSectionLookup = (from e in _dbAhobpr.STD_FORM_SUB_SECTIONs
                                                             select e).ToList();
                    }
                }

                int subSectionCount = (from e in AHOBPRGlobal.FormSubSectionLookup
                                       where e.STD_FORM_SECTION_ID == sqlSectionId
                                       select e).Count();

                result = (subSectionCount > 1);
            }

            return result;
        }

        /// <summary>
        /// Get STD_FORM_ANSWER_ID by sqlQuestionId and bprAnswerId
        /// </summary>
        /// <param name="sqlQuestionId"></param>
        /// <param name="bprAnswerId"></param>
        /// <returns></returns>
        public int GetFormAnswerId(int sqlQuestionId, string bprAnswerId)
        {
            int id = 0;
            if (sqlQuestionId == 0 || string.IsNullOrEmpty(bprAnswerId))
            {
                return id;
            }
            else
            {
                if (AHOBPRGlobal.FormAnswerLookup == null)
                {
                    using (_dbAhobpr = GetDataContext())
                    {
                        AHOBPRGlobal.FormAnswerLookup = (from e in _dbAhobpr.STD_FORM_ANSWERs
                                                         select e).ToList();
                    }
                }

                try
                {
                    id = (from e in AHOBPRGlobal.FormAnswerLookup
                          where e.STD_FORM_QUESTION_ID == sqlQuestionId && e.BRP_FORM_ANSWER_ID == bprAnswerId
                          select e.STD_FORM_ANSWER_ID).FirstOrDefault();
                }
                catch
                {
                    id = 0;
                }

                return id;
            }
        }

        public String GetFormAnswer(int sqlQuestionId, string bprAnswerId)
        {
            String answer = "";
            if (sqlQuestionId == 0 || string.IsNullOrEmpty(bprAnswerId))
            {
                return answer;
            }
            else
            {
                if (AHOBPRGlobal.FormAnswerLookup == null)
                {
                    using (_dbAhobpr = GetDataContext())
                    {
                        AHOBPRGlobal.FormAnswerLookup = (from e in _dbAhobpr.STD_FORM_ANSWERs
                                                         select e).ToList();
                    }
                }

                try
                {
                    answer = (from e in AHOBPRGlobal.FormAnswerLookup
                          where e.STD_FORM_QUESTION_ID == sqlQuestionId && e.BRP_FORM_ANSWER_ID == bprAnswerId
                          select e.TEXT).FirstOrDefault();
                }
                catch
                {
                    answer = "";
                }

                return answer;
            }
        }

      #endregion

        public STD_FORM GetFormByBprId(String bprId)
        {
            STD_FORM form = null;
            using (_dbAhobpr = GetDataContext())
            {
                SetLoadWithForForms(_dbAhobpr);
                form = (from e in _dbAhobpr.STD_FORMs
                        where e.BPR_FORM_ID == bprId
                        select e).FirstOrDefault();

            }
            return form;
        }

        public STD_FORM GetFormByToolFormId(string toolFormId)
        {
            STD_FORM form = null;
            if (string.IsNullOrEmpty(toolFormId))
            {
                return form;
            }
            else
            {
                int id = Convert.ToInt32(toolFormId);
                using (_dbAhobpr = GetDataContext())
                {
                    SetLoadWithForForms(_dbAhobpr);
                    form = (from e in _dbAhobpr.STD_FORMs
                            where e.TOOL_FORM_ID == id
                            select e).FirstOrDefault();

                }
            }
            return form;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="formId"></param>
        /// <returns></returns>
        [System.Diagnostics.CodeAnalysis.SuppressMessage("Microsoft.Naming", "CA1702:CompoundWordsShouldBeCasedCorrectly", MessageId = "FormBy")]
        public STD_FORM GetFormByStdFormId(string formId)
        {
            STD_FORM form = null;
            if (string.IsNullOrEmpty(formId))
            {
                return form;
            }
            else
            {
                int id = Convert.ToInt32(formId);
                using (_dbAhobpr = GetDataContext())
                {
                    form = (from e in _dbAhobpr.STD_FORMs
                            where e.STD_FORM_ID == id
                            select e).FirstOrDefault();

                }
            }
            return form;
        }

        /// <summary>
        /// Check if a form is a system wide follow-up questionnaire
        /// </summary>
        /// <param name="formId"></param>
        /// <returns></returns>
        public bool IsFormSystemWide(int formId)
        {
            bool isSystemWide = false;

            using (_dbAhobpr = GetDataContext())
            {
                int? toolFormId = (from e in _dbAhobpr.STD_FORMs
                        where e.STD_FORM_ID == formId 
                        select e.TOOL_FORM_ID).FirstOrDefault();

                isSystemWide = (from e in _dbAhobpr.TOOL_FORMs
                                where e.SYSTEM_WIDE_FLAG == true
                                   && e.TOOL_FORM_ID == toolFormId
                                select e).ToList().Count > 0;
            }
            return isSystemWide ;
        }

        /// <summary>
        /// update form
        /// </summary>
        /// <param name="form"></param>
        /// <returns></returns>
        public int UpdateForm(STD_FORM form)
        {
            int returnStatus = 0;

            if (form != null)
            {
                using (_dbAhobpr = GetDataContext())
                {
                    _dbAhobpr.DeferredLoadingEnabled = false;

                    //this line traverses all entities, attaching all of them as appropriate to the data context.
                    form.SynchroniseWithDataContext(_dbAhobpr);

                    //Check if any actual changes will occur
                    ChangeSet changeSet = _dbAhobpr.GetChangeSet();

                    if (changeSet.Updates.Count > 0 || changeSet.Inserts.Count > 0)
                    {
                        //if changes present then submit changes
                        try
                        {
                            _dbAhobpr.SubmitChanges(ConflictMode.ContinueOnConflict);
                            returnStatus = 1;

                        }
                        catch (System.Data.SqlClient.SqlException ex)
                        {
                            _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                            returnStatus = -1;
                        }
                        catch (ChangeConflictException)
                        {
                            _dbAhobpr.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);

                            returnStatus = 1;
                        }
                        catch (Exception ex)
                        {
                            _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                            returnStatus = -1;
                        }
                    }
                }
            }
            ResetGlobalVariables();
            return returnStatus;
        }

        public void SetLoadWithForForms(AHOBPRDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();

            lo.LoadWith<STD_FORM>(e => e.STD_FORM_SECTIONs);
            lo.LoadWith<STD_FORM_SECTION>(e => e.STD_FORM_SUB_SECTIONs);
            lo.LoadWith<STD_FORM_SUB_SECTION>(e => e.STD_FORM_QUESTIONs);
            lo.LoadWith<STD_FORM_QUESTION>(e => e.STD_FORM_ANSWERs);
            lo.LoadWith<STD_FORM_ANSWER>(e => e.STD_FORM_ANSWER_FIELDs);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }
    }
}
